package gov.va.med.edp.springframework.security.providers.vistalink;

import gov.va.med.edp.springframework.security.userdetails.VistaUserDetails;
import gov.va.med.edp.springframework.security.providers.vistalink.VistaAuthenticationToken;
import junit.framework.TestCase;
import org.springframework.security.GrantedAuthority;
import org.easymock.MockControl;

public class VistaAuthenticationTokenTest extends TestCase {

    public void testConstruct() {
        VistaAuthenticationToken token = new VistaAuthenticationToken("891", "10VEHU", "VEHU10", "127.0.0.1");
        assertNull(token.getDuz());
        assertEquals("891", token.getStationNumber());
        assertEquals("10VEHU", token.getAccessCode());
        assertEquals("VEHU10", token.getVerifyCode());
        assertEquals("127.0.0.1", token.getRemoteAddress());

        assertEquals(VistaAuthenticationToken.UNAUTHENTICATED + "@891", token.getPrincipal());
        assertEquals("10VEHU;VEHU10;127.0.0.1", token.getCredentials());
        assertEquals("127.0.0.1", token.getDetails());
        assertFalse(token.isAuthenticated());
    }

    public void testAuthenticatedConstruct() {
        MockControl userControl = MockControl.createControl(VistaUserDetails.class);
        VistaUserDetails user = (VistaUserDetails) userControl.getMock();
        userControl.expectAndDefaultReturn(user.getSignonLogInternalEntryNumber(), "12345.987654321");
        userControl.expectAndDefaultReturn(user.getLoginStationNumber(), "891");
        userControl.expectAndDefaultReturn(user.getDuz(), "101284");
        userControl.replay();

        VistaAuthenticationToken token = new VistaAuthenticationToken(user, "10VEHU", "VEHU10", "127.0.0.1", new GrantedAuthority[]{});
        assertEquals(user.getDuz(), token.getDuz());
        assertEquals(user.getLoginStationNumber(), token.getStationNumber());
        assertEquals("10VEHU", token.getAccessCode());
        assertEquals("VEHU10", token.getVerifyCode());
        assertEquals("127.0.0.1", token.getRemoteAddress());

        assertSame(user, token.getPrincipal());
        assertEquals("10VEHU;VEHU10;127.0.0.1", token.getCredentials());
        assertEquals("127.0.0.1", token.getDetails());
        assertTrue(token.isAuthenticated());
    }

// this is the java 5 easymock 2 version
//    public void testAuthenticatedConstruct() {
//        VistaUserDetails user = EasyMock.createMock(VistaUserDetails.class);
//        EasyMock.expect(user.getSignonLogInternalEntryNumber()).andReturn("12345.987654321").anyTimes();
//        EasyMock.expect(user.getLoginStationNumber()).andReturn("891").anyTimes();
//        EasyMock.expect(user.getDuz()).andReturn("101284").anyTimes();
//        EasyMock.replay(user);
//
//        VistaAuthenticationToken token = new VistaAuthenticationToken(user, "10VEHU", "VEHU10", "127.0.0.1", new GrantedAuthority[]{});
//        assertEquals(user.getDuz(), token.getDuz());
//        assertEquals(user.getLoginStationNumber(), token.getStationNumber());
//        assertEquals("10VEHU", token.getAccessCode());
//        assertEquals("VEHU10", token.getVerifyCode());
//        assertEquals("127.0.0.1", token.getRemoteAddress());
//
//        assertSame(user, token.getPrincipal());
//        assertEquals("10VEHU;VEHU10;127.0.0.1", token.getCredentials());
//        assertEquals("127.0.0.1", token.getDetails());
//        assertTrue(token.isAuthenticated());
//    }


    public void testConstructWithMissingAccessCode() {
        VistaAuthenticationToken token = new VistaAuthenticationToken("891", null, "VEHU10", "127.0.0.1");
        assertNull(token.getCredentials());
        assertEquals("VEHU10", token.getVerifyCode());
        assertEquals("127.0.0.1", token.getRemoteAddress());
        token = new VistaAuthenticationToken("891", "", "VEHU10", "127.0.0.1");
        assertNull(token.getCredentials());
        assertEquals("VEHU10", token.getVerifyCode());
        assertEquals("127.0.0.1", token.getRemoteAddress());
    }

    public void testConstructWithMissingVerifyCode() {
        VistaAuthenticationToken token = new VistaAuthenticationToken("891", "10VEHU", null, "127.0.0.1");
        assertNull(token.getCredentials());
        assertEquals("10VEHU", token.getAccessCode());
        assertEquals("127.0.0.1", token.getRemoteAddress());
        token = new VistaAuthenticationToken("891", "10VEHU", "", "127.0.0.1");
        assertNull(token.getCredentials());
        assertEquals("10VEHU", token.getAccessCode());
        assertEquals("127.0.0.1", token.getRemoteAddress());
    }

    public void testConstructWithMissingRemoteAddress() {
        VistaAuthenticationToken token = new VistaAuthenticationToken("891", "10VEHU", "VEHU10", null);
        assertNull(token.getCredentials());
        assertEquals("10VEHU", token.getAccessCode());
        assertEquals("VEHU10", token.getVerifyCode());
        token = new VistaAuthenticationToken("891", "10VEHU", "VEHU10", "");
        assertNull(token.getCredentials());
        assertEquals("10VEHU", token.getAccessCode());
        assertEquals("VEHU10", token.getVerifyCode());
    }

    public void testConstructWithMissingCredentials() {
        VistaAuthenticationToken token = new VistaAuthenticationToken("891", null, null, null);
        assertNull(token.getCredentials());
        assertNull(token.getAccessCode());
        assertNull(token.getVerifyCode());
        assertNull(token.getRemoteAddress());
        token = new VistaAuthenticationToken("891", "", "", "");
        assertNull(token.getCredentials());
        assertNull(token.getAccessCode());
        assertNull(token.getVerifyCode());
        assertNull(token.getRemoteAddress());
    }

    public void testSetDetailsIsNoop() {

    }
}
