package gov.va.med.edp.dao.rpc;

import gov.va.med.edp.dao.TrackingDao;
import gov.va.med.edp.vistalink.VistaLinkDaoSupport;
import org.springframework.dao.DataAccessException;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class VistaLinkBigBoardDao extends VistaLinkDaoSupport implements TrackingDao {
    static final String EDPS_BOARD_CONTEXT = "EDPS BOARD CONTEXT";
    static final String EDP_CONTROLLER_BOARD_RPC = "EDPCBRD RPC";
    static final String EDPTRACKING_APPLICATION_USER = "EDPTRACKING,PROXY";

    /**
     * Builds the list of parameters to pass to the RPC.
     * First argument to the RPC is a string of the format "${duz}^${stationNumber}".  If duz is null, will send "0^${stationNumber}"
     * Second argument to the RPC is a multiple containing key value pairs passed in as a map.
     *
     * @param stationNumber The station number that this command is being executed on behalf of.  Note how this is different
     *                      from the station number of the vista account this dao is communicating with.
     * @param duz           user who command is being executed on behalf of.  Can be null for some operations. Note how this is different
     *                      from the "application user duz" that RPCs are executed with.
     * @param params        A map containing key value pairs that are passed to RPC as a multiple.
     * @return list with two items in it, a string and a multiple, to be passed as RPC arguments.
     */
    protected List buildRpcParameterList(String stationNumber, String duz, Map params) {
        ArrayList paramList = new ArrayList();
        paramList.add((duz == null ? "0" : duz) + '^' + stationNumber);
        paramList.add(params);
        return paramList;
    }

    /**
     * Commands executed by this data access object are always executed by the "application user" against the stationNumber
     * set by the property on this bean.
     *
     * @param stationNumber The station number that this command is being executed on behalf of. Note how this is different
     *                      from the station number of the vista account this dao is communicating with.  Cannot be null.
     * @param duz           A duz to execute the command on behalf of. Can be null for some operations. Note how this is different
     *                      from the "application user duz" that RPCs are executed with.
     * @param params        A map containing key value pairs that are passed to the RPC as a multiple.
     * @return result of executing command RPC as a string.
     * @throws org.springframework.dao.DataAccessException
     *
     */
    public String executeCommand(String stationNumber, String duz, Map params) throws DataAccessException {
        return getRpcTemplate().rpcAsApplication(stationNumber, EDPTRACKING_APPLICATION_USER, EDPS_BOARD_CONTEXT, EDP_CONTROLLER_BOARD_RPC, buildRpcParameterList(stationNumber, duz, params));
    }
}
