unit ShellLinker;

interface

uses
  Windows, ShlObj;

type
  TShowState = (ssNormal, ssMinimized, ssMaximized);

  TLinkInfo = record
    Target: String;          // Path to link target file   - empty for virtual folders
    ClsID: PItemIDList;      // Absolute CLSID (PIDL) of target
    WorkDir: String;         // Working directory          - empty for virtual folders
    Parameters: String;      // Parameters sent to path    - empty for virtual folders
    Description: String;     // Description of link
    IconPath: String;        // File containing icon       - often same as Target
    IconIndex: Integer;      // Index of icon in the icon file
    HotKey: Word;            // HotKey (like Alt+Shift+F)  - numeric
    ShowState: TShowState;   // Normal, minimized, or maximized
  end;

  TLinkHotKey = record
    Modifiers: Word;
    Key: Word;
  end;

  function CreateLink(Title, Folder, Target, Parameters, Description: String): Boolean;
  function CreateLinkEx(Title, Folder: String; LinkInfo: TLinkInfo): Boolean;
  function GetLinkInfo(FileName: String; var LinkInfo: TLinkInfo): Boolean;
  function ExecuteLink(FileName: String; Handle: HWND; Verb: PChar): Boolean; overload;
  function ExecuteLink(FileName: String; Handle: HWND; Verb: PChar;
    ForceRestore: Boolean): Boolean; overload;
  function GetTargetDisplayName(ClsID: PItemIDList): String;
  function GetTargetIconHandle(ClsID: PItemIDList; SmallIcon, LinkOverlay: Boolean): Cardinal;
  function GetSpecialFolderPath(FolderIndex: Integer): String;
  function LinkHotKeyToWord(Modifiers, Key: Word): Word;
  function WordToLinkHotKey(HotKey: Word): TLinkHotKey;
  function TranslateHotKeyToText(HotKey: Word): String;


implementation

uses
  SysUtils, ComObj, ActiveX, ShellApi, Classes, Menus;


function ShowStateToCmdShow(ShowState: TShowState): Integer;
begin
  case ShowState of
    ssMinimized: Result := SW_SHOWMINNOACTIVE;
    ssMaximized: Result := SW_SHOWMAXIMIZED;
  else
    Result := SW_SHOWNORMAL;
  end;
end;


function CmdShowToShowState(Message: Integer): TShowState;
begin
  case Message of
    SW_SHOWMINNOACTIVE, SW_SHOWMINIMIZED: Result := ssMinimized;
    SW_SHOWMAXIMIZED:                     Result := ssMaximized;
  else
    Result := ssNormal;
  end;
end;


function CreateShellLink(FileName, Target, Description, StartIn, Parameters,
  IconPath: String; IconIndex: Integer; HotKey: Word; ShowState: TShowState;
  ItemIDList: PItemIDList): Boolean;
var
  sl: IShellLink;
  ppf: IPersistFile;
  wcLinkName: array[0..MAX_PATH] of WideChar;
begin
  CoInitialize(nil);         // Initialize COM
  CoCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLinkA, sl);
  sl.QueryInterface(IPersistFile, ppf);
  sl.SetPath(PChar(Target));
  if ItemIDList <> nil then
    sl.SetIDList(ItemIDList);
  sl.SetDescription(PChar(Description));
  sl.SetWorkingDirectory(PChar(StartIn));
  sl.SetArguments(PChar(Parameters));
  sl.SetIconLocation(PChar(IconPath), IconIndex);
  if HotKey <> 0 then
    sl.SetHotkey(HotKey);
  sl.SetShowCmd(ShowStateToCmdShow(ShowState));
  // Save shell link
  MultiByteToWideChar(CP_ACP, 0, PChar(FileName), -1, wcLinkName, MAX_PATH);
  Result := (ppf.Save(wcLinkName, true) = S_OK);
  CoUninitialize;
end;


function StrRetToStr(ItemIDList: PItemIDList; StrRet: TStrRet): String;
var
  P: PChar;
begin
  case StrRet.uType of
    STRRET_WSTR: begin             // UNICODE
      Result := StrRet.pOleStr;
//      Malloc.Free(StrRet.pOleStr);        // Should we deallocate?
    end;
    STRRET_CSTR:                   // ANSI
      Result := StrRet.cStr;
    STRRET_OFFSET: begin
      P := @ItemIDList.mkid.abID[StrRet.uOffset - SizeOf(ItemIDList.mkid.cb)];
      SetString(Result, P, ItemIDList.mkid.cb - StrRet.uOffset);
    end;
  end;
end;


function CreateLink(Title, Folder, Target, Parameters, Description: String): Boolean;
begin
  Result := CreateShellLink(Folder +'\' + Title + '.lnk', Target, Description,
                  ExtractFilePath(Target), Parameters, Target, 0, 0, ssNormal, nil);
end;


function CreateLinkEx(Title, Folder: String; LinkInfo: TLinkInfo): Boolean;
begin
  with LinkInfo do
   Result :=  CreateShellLink(Folder +'\' + Title + '.lnk', Target, Description,
                    WorkDir, Parameters, IconPath, IconIndex, HotKey, ShowState, ClsID);
end;


function GetLinkInfo(FileName: String; var LinkInfo: TLinkInfo): Boolean;
var
  sl: IShellLink;
  ppf: IPersistFile;
  wcLinkName: array[0..MAX_PATH] of WideChar;
  sTarget, sDir, sParams, sDesc, sIcon: String;
  sClsID: PItemIDList;
  sHotKey: Word;
  sCmdShow: Integer;
  Idx: Integer;
  wfd: TWin32FindData;
begin
  Result := True;
  FillChar(LinkInfo, SizeOf(LinkInfo), 0);
  SetLength(sDesc, MAX_PATH);
  SetLength(sTarget, MAX_PATH);
  SetLength(sDir, MAX_PATH);
  SetLength(sParams, MAX_PATH);
  SetLength(sIcon, MAX_PATH);
  sClsID := nil;
  CoInitialize(nil);         // Initialize COM

  try
    CoCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLinkA, sl);
    sl.QueryInterface(IPersistFile, ppf);
    MultiByteToWideChar(CP_ACP, 0, PChar(FileName), -1, wcLinkName, MAX_PATH);
    ppf.Load(wcLinkName, STGM_READ);
    with sl do
    begin
      GetPath(PChar(sTarget), MAX_PATH, wfd, SLR_ANY_MATCH);
      GetIDList(sClsID);
      GetDescription(PChar(sDesc), MAX_PATH);
      GetWorkingDirectory(PChar(sDir), MAX_PATH);
      GetArguments(PChar(sParams), MAX_PATH);
      GetIconLocation(PChar(sIcon), MAX_PATH, Idx);
      GetHotkey(sHotKey);
      GetShowCmd(sCmdShow);
    end;
  except
    on EOleSysError do
      Result := False;
  end;

  CoUninitialize;

  if sClsID = nil then
    Result := False
  else
  begin
    SetLength(sTarget, StrLen(PChar(sTarget)));
    SetLength(sDir, StrLen(PChar(sDir)));
    SetLength(sDesc, StrLen(PChar(sDesc)));
    SetLength(sParams, StrLen(PChar(sParams)));
    SetLength(sIcon, StrLen(PChar(sIcon)));
    LinkInfo.Target := sTarget;
    // Target is empty if the link references a virtual folder, like My Computer
    LinkInfo.ClsID := sClsID;     // ClsID is always absolute
    LinkInfo.WorkDir  := sDir;
    LinkInfo.Description := sDesc;
    LinkInfo.Parameters := sParams;
    if StrLen(PChar(sIcon)) <> 0 then
      LinkInfo.IconPath := sIcon
    else
      LinkInfo.IconPath := sTarget;
    LinkInfo.IconIndex := Idx;
    LinkInfo.HotKey := sHotKey;
    LinkInfo.ShowState := CmdShowToShowState(sCmdShow);
  end;
end;


function ExecuteLink(FileName: String; Handle: HWND; Verb: PChar): Boolean;
// Execute the selected link
begin
  Result := ExecuteLink(FileName, Handle, Verb, False);
end;


function ExecuteLink(FileName: String; Handle: HWND; Verb: PChar;
  ForceRestore: Boolean): Boolean;
// Execute the selected link
var
  li: TLinkInfo;
  show: Integer;
  fn: String;
begin
  GetLinkInfo(FileName, li);
  if li.Target <> '' then
    fn := li.Target          // Execute link's target
  else
    fn := FileName;          // Execute link
  if ForceRestore then
    show := SW_RESTORE
  else
    show := ShowStateToCmdShow(li.ShowState);
  Result := (ShellExecute(Handle, Verb, PChar(fn), nil, '.', show) > 32);
  // Use GetLastError to return error code if ShellExecute failed
end;


function GetTargetDisplayName(ClsID: PItemIDList): String;
var
  shf: IShellFolder;
  StrRet: TStrRet;
begin
  CoInitialize(nil);
  try
    SHGetDesktopFolder(shf);
    shf.GetDisplayNameOf(ClsID, {SHGDN_NORMAL} SHGDN_FORADDRESSBAR or SHGDN_FORPARSING, StrRet);
    Result := StrRetToStr(ClsID, StrRet);
  except
    on Exception do
      Result := '';
  end;
  CoUninitialize;
end;


function GetSpecialFolderPath(FolderIndex: Integer): String;
var
  iID: PItemIDList;
  szPath: PChar;
begin
  szPath := StrAlloc(MAX_PATH);
  if (SHGetSpecialFolderLocation(0, FolderIndex, iID) = NOERROR) then
  begin
    SHGetPathFromIDList(iID, szPath);
    Result := szPath;
  end
  else
    Result := '';
  StrDispose(szPath);
end;


function GetTargetIconHandle(ClsID: PItemIDList; SmallIcon, LinkOverlay: Boolean): Cardinal;
var
  iml: Cardinal;   // HIMAGELIST
  shfi: TSHFileInfo;
  flags: Cardinal;
begin
  Result := 0;
  FillChar(shfi, SizeOf(shfi), 0);
  flags := SHGFI_PIDL {+ SHGFI_SYSICONINDEX} + SHGFI_ICON;
  if SmallIcon then
    flags := flags + SHGFI_SMALLICON;
  if LinkOverlay then
    flags := flags + SHGFI_LINKOVERLAY;
  iml := SHGetFileInfo(PChar(ClsID), 0, shfi, SizeOf(shfi), flags);
  if (shfi.hIcon <> 0) and (iml <> 0) then
    Result := shfi.hIcon;
end;


function LinkHotKeyToWord(Modifiers, Key: Word): Word;
begin
  Result := (Modifiers shl 8) + Key;
end;


function WordToLinkHotKey(HotKey: Word): TLinkHotKey;
begin
  Result.Modifiers := HotKey shr 8;
  Result.Key := LoByte(HotKey);
end;


function TranslateHotKeyToText(HotKey: Word): String;
{ The hotkey received from IShellLink.GetHotkey must be manipulated to get
  a correct translation to text. BTW: The hotkey is NOT in the same format as
  hotkeys used with the WM_HOTKEY message.
  We modify the hotkey so it can be used in a call to Menus.ShortCutToText. }
var
  Modifiers: Word;
  shk: TLinkHotKey;
begin
  shk := WordToLinkHotKey(HotKey);
  Modifiers := 0;

  // Switch Alt and Shift
  if (shk.Modifiers and MOD_ALT) <> 0 then       // MOD_ALT = HOTKEYF_SHIFT
    Inc(Modifiers, scShift);
  if (shk.Modifiers and MOD_SHIFT) <> 0 then     // MOD_SHIFT = HOTKEYF_ALT
    Inc(Modifiers, scAlt);

  if (shk.Modifiers and MOD_CONTROL) <> 0 then   // MOD_CONTROL = HOTKEYF_CONTROL
    Inc(Modifiers, scCtrl);
  Result := ShortCutToText(Modifiers + LoByte(HotKey));
end;

end.

