unit fPtDemoEdit;
//kt Added this entire unit for demographics editing at runtime.

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, ExtCtrls, DKLang;

type
  BoolUC = (bucFalse, bucTrue, bucUnchanged);
  
  TPatientInfo = class(TObject)
  public
    LName: String;
    FName: String;
    MName: String;
    CombinedName: String;
    Prefix: String;
    Suffix: String;
    Degree: String;
    DOB: String;
    Sex: String;
    SSNum: String;
    EMail: String;
    AliasInfo : TStringList;  //format: s=IEN#, Object is ^tAlias
    AddressLine1: String;
    AddressLine2: String;
    AddressLine3: String;
    City: String;
    State: String;
    Zip4: String;
    BadAddress: BoolUC;
    PhoneNumResidence: String;
    PhoneNumWork: String;
    PhoneNumCell: String;
    PhoneNumTemp: String;

    TempAddressLine1: String;
    TempAddressLine2: String;
    TempAddressLine3: String;
    TempCity: String;
    TempState: String;
    TempZip4: String;
    TempStartingDate : String;
    TempEndingDate : String;
    TempAddressActive: BoolUC;
      
    ConfidentalAddressLine1: String;
    ConfidentalAddressLine2: String;
    ConfidentalAddressLine3: String;
    ConfidentalCity: String;
    ConfidentalState: String;
    ConfidentalZip: String;
    ConfidentalStartingDate : String;
    ConfidentalEndingDate : String;
    ConfAddressActive : BoolUC;

    Modified : boolean;
      
    constructor Create;
    destructor Destroy; override;
    procedure ClearAliasInfo;
    procedure Clear;
    procedure Assign(Source : TPatientInfo);
    procedure RemoveUnchanged(OldInfo : TPatientInfo);
  end;

  TfrmPtDemoEdit = class(TForm)
    OKBtn: TButton;
    CancelBtn: TButton;
    ApplyBtn: TButton;
    PageControl: TPageControl;
    DemoTabSheet: TTabSheet;
    LNameLabel: TLabel;
    FNameLabel: TLabel;
    MNameLabel: TLabel;
    CombinedNameLabel: TLabel;
    PrefixLabel: TLabel;
    SuffixLabel: TLabel;
    DOBLabel: TLabel;
    SSNumLabel: TLabel;
    CombinedNameEdit: TEdit;
    LNameEdit: TEdit;
    FNameEdit: TEdit;
    MNameEdit: TEdit;
    PrefixEdit: TEdit;
    SuffixEdit: TEdit;
    DOBEdit: TEdit;
    SSNumEdit: TEdit;
    AliasGroupBox: TGroupBox;
    AliasComboBox: TComboBox;
    AliasNameLabel: TLabel;
    AliasSSNumLabel: TLabel;
    AliasNameEdit: TEdit;
    AliasSSNEdit: TEdit;
    AddressGroupBox: TGroupBox;
    AddressRGrp: TRadioGroup;
    AddressLine1Edit: TEdit;
    AddressLine2Edit: TEdit;
    AddressLine3Edit: TEdit;
    CityLabel: TLabel;
    CityEdit: TEdit;
    StateComboBox: TComboBox;
    Zip4Edit: TEdit;
    Zip4Label: TLabel;
    BadAddressCB: TCheckBox;
    TempActiveCB: TCheckBox;
    SexLabel: TLabel;
    SexComboBox: TComboBox;
    DelAliasBtn: TButton;
    StartingDateEdit: TEdit;
    StartingDateLabel: TLabel;
    EndingDateLabel: TLabel;
    EndingDateEdit: TEdit;
    ConfActiveCB: TCheckBox;
    DegreeEdit: TEdit;
    DegreeLabel: TLabel;
    AddAliasBtn: TButton;
    GroupBox1: TGroupBox;
    PhoneNumGrp: TRadioGroup;
    PhoneNumEdit: TEdit;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    DKLanguageController1: TDKLanguageController;
    EMailEdit: TEdit;
    procedure AliasComboBoxChange(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure AddressRGrpClick(Sender: TObject);
    procedure PhoneNumGrpClick(Sender: TObject);
    procedure DelAliasBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure CombinedNameEditChange(Sender: TObject);
    procedure LNameEditChange(Sender: TObject);
    procedure FNameEditChange(Sender: TObject);
    procedure MNameEditChange(Sender: TObject);
    procedure PrefixEditChange(Sender: TObject);
    procedure SuffixEditChange(Sender: TObject);
    procedure SexComboBoxChange(Sender: TObject);
    procedure EMailEditChange(Sender: TObject);
    procedure AddressLine1EditChange(Sender: TObject);
    procedure AddressLine2EditChange(Sender: TObject);
    procedure AddressLine3EditChange(Sender: TObject);
    procedure CityEditChange(Sender: TObject);
    procedure Zip4EditChange(Sender: TObject);
    procedure StateComboBoxChange(Sender: TObject);
    procedure TempActiveCBClick(Sender: TObject);
    procedure ConfActiveCBClick(Sender: TObject);
    procedure BadAddressCBClick(Sender: TObject);
    procedure DegreeEditChange(Sender: TObject);
    procedure PhoneNumEditChange(Sender: TObject);
    procedure StartingDateEditChange(Sender: TObject);
    procedure EndingDateEditChange(Sender: TObject);
    procedure AliasNameEditChange(Sender: TObject);
    procedure AliasSSNEditChange(Sender: TObject);
    procedure ApplyBtnClick(Sender: TObject);
    procedure AddAliasBtnClick(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure CancelBtnClick(Sender: TObject);
    procedure DOBEditChange(Sender: TObject);
    procedure SSNumEditChange(Sender: TObject);
  private
    { Private declarations }
    FCurPatientInfo : TPatientInfo;
    FServerPatientInfo : TPatientInfo;
    FCurAliasEdit : integer;
    ProgAliasChangeOccuring : boolean;
    ProgNameChangeOccuring : boolean;
    ProgPhoneChangeOccuring : boolean;
    ProgAddressChangeOccuring : boolean;
    MaxAliasIEN : integer;
    ChangesMade : boolean;
    procedure GetPtInfo(PatientInfo : TPatientInfo);
    procedure PostChangedInfo(PatientInfo : TPatientInfo);
    procedure ShowAliasInfo(Patient : TPatientInfo);
    procedure ShowPtInfo(Patient : TPatientInfo);
    function CombinedName : string;
    procedure NameParts(CombinedName: string; var LName, FName, MName : string);
    procedure SetModified(value : boolean);
    procedure SetAliasEnabled(value : boolean);
  public
    { Public declarations }
  end;

var
  frmPtDemoEdit: TfrmPtDemoEdit;

implementation

{$R *.dfm}

uses
  IniFiles,Trpcb,ORNet,uCore, mfunstr, strutils;

const
  ADD_NEW_ALIAS = '<Add New Alias>';

type
  tAlias= class
    Name : string;
    SSN : string;
    procedure Assign(Source: tAlias);
  end;

procedure tAlias.Assign(Source : tAlias);
begin
  Name := Source.Name;
  SSN := Source.SSN;
end;  

//=========================================================
//=========================================================
//=========================================================
  
constructor TPatientInfo.Create;
begin
  AliasInfo := TStringList.Create;
  Clear;
end;

destructor TPatientInfo.Destroy;
begin
  ClearAliasInfo;  
  AliasInfo.Free;
  inherited Destroy;
End;

procedure TPatientInfo.ClearAliasInfo;
var i : integer;
    pAlias : tAlias;
begin
  for i := 0 to AliasInfo.Count-1 do begin
    pAlias := tAlias(AliasInfo.Objects[i]);
    pAlias.Free;
  end;
  AliasInfo.Clear
End;

procedure TPatientInfo.Clear;
begin
  LName:= '';
  FName:= '';
  MName:= '';
  CombinedName:= '';
  Prefix:= '';
  Suffix:= '';
  Degree:= '';
  DOB:= '';
  SSNum:= '';
  ClearAliasInfo;  
  AddressLine1:= '';
  AddressLine2:= '';
  AddressLine3:= '';
  City:= '';
  State:= '';
  Zip4:= '';
  TempAddressLine1:= '';
  TempAddressLine2:= '';
  TempAddressLine3:= '';
  TempCity:= '';
  TempState:= '';
  TempZip4:= '';
  TempStartingDate := '';
  TempEndingDate := '';
  ConfidentalAddressLine1:= '';
  ConfidentalAddressLine2:= '';
  ConfidentalAddressLine3:= '';
  ConfidentalCity:= '';
  ConfidentalState:= '';
  ConfidentalZip:= '';
  ConfidentalStartingDate := '';
  ConfidentalEndingDate := '';
  BadAddress:= bucFalse;
  TempAddressActive:= bucFalse;
  ConfAddressActive := bucFalse;
  Modified := false;
  
  PhoneNumResidence:= '';
  PhoneNumWork:= '';
  PhoneNumCell:= '';
  PhoneNumTemp:= '';
  Sex:= '';
end;


procedure TPatientInfo.Assign(Source : TPatientInfo);
var i : integer;
    pAlias : tAlias;
    OtherpAlias : tAlias;
begin
  LName:=Source.LName;
  FName:=Source.FName;
  MName:=Source.MName;
  CombinedName:=Source.CombinedName;
  Prefix:=Source.Prefix;
  Suffix:=Source.Suffix;
  Degree:=Source.Degree;
  DOB:=Source.DOB;
  SSNum:=Source.SSNum;
  
  ClearAliasInfo;
  //Copy pointed to tAlias entries, don't simply copy references
  for i := 0 to Source.AliasInfo.Count-1 do begin
    AliasInfo.Add(Source.AliasInfo.Strings[i]);
    OtherpAlias := tAlias(Source.AliasInfo.Objects[i]);
    if OtherpAlias<>nil then begin
      pAlias := tAlias.Create;
      pAlias.Name := OtherpAlias.Name;
      pAlias.SSN := OtherpAlias.SSN;
      AliasInfo.Objects[i]:=pAlias;    
    end;
  end;
  AddressLine1:=Source.AddressLine1;
  AddressLine2:=Source.AddressLine2;
  AddressLine3:=Source.AddressLine3;
  City:=Source.City;
  State:=Source.State;
  Zip4:=Source.Zip4;
  TempAddressLine1:=Source.TempAddressLine1;
  TempAddressLine2:=Source.TempAddressLine2;
  TempAddressLine3:=Source.TempAddressLine3;
  TempCity:=Source.TempCity;
  TempState:=Source.TempState;
  TempZip4:=Source.TempZip4;
  TempStartingDate :=Source.TempStartingDate ;
  TempEndingDate :=Source.TempEndingDate ;
  ConfidentalAddressLine1:=Source.ConfidentalAddressLine1;
  ConfidentalAddressLine2:=Source.ConfidentalAddressLine2;
  ConfidentalAddressLine3:=Source.ConfidentalAddressLine3;
  ConfidentalCity:=Source.ConfidentalCity;
  ConfidentalState:=Source.ConfidentalState;
  ConfidentalZip:=Source.ConfidentalZip;
  ConfidentalStartingDate :=Source.ConfidentalStartingDate ;
  ConfidentalEndingDate :=Source.ConfidentalEndingDate ;
  BadAddress:= Source.BadAddress;
  TempAddressActive:= Source.TempAddressActive;
  ConfAddressActive := Source.ConfAddressActive;
  PhoneNumResidence:=Source.PhoneNumResidence;
  PhoneNumWork:=Source.PhoneNumWork;
  PhoneNumCell:=Source.PhoneNumCell;
  PhoneNumTemp:=Source.PhoneNumTemp;
  Sex:=Source.Sex;
  EMail := Source.EMail;
end;


procedure TPatientInfo.RemoveUnchanged(OldInfo : TPatientInfo);
//Will remove entries that are unchanged from OldInfo
//ALSO, will change AliasInfo entries:
//     Other code adds "IEN" numbers that don't have any corresponding
//     true IEN on the server.  This will convert these to +1,+2 etc.
//     And, if there is an alias entry in the OldInfo that is not
//     in this info, then a matching @ entry for that IEN will be generated.

  procedure CompStrs(var newS, oldS : string);
  begin
    if newS = oldS then begin
      newS := '';  //no change, 
    end else begin
      if (newS = '') and (oldS <> '') then newS := '@' //delete symbol
    end;
  end;

  procedure CompBoolUC(var newBN, oldBN : BoolUC);
  begin
    if newBN=oldBN then begin
      newBN := bucUnchanged;  //Mark unchanged
    end;  
  end;

  const
    BOOL_STR : array[false..true] of string =('TRUE','FALSE');
    NO_CHANGE = 1;
    NEW_RECORD = 2;
    DELETED_RECORD = 3;
    CHANGED_RECORD = 4;

  function CompAliasRec(curAlias,oldAlias : tAlias) : integer;
  //Returns: NO_CHANGE = 1; NEW_RECORD = 2; DELETED_RECORD = 3; CHANGED_RECORD = 4;
  begin
    Result := NO_CHANGE;
    if (curAlias <> nil) and (oldAlias <> nil) then begin
      if curAlias.Name = '' then begin 
        if oldAlias.Name <> '' then Result := DELETED_RECORD;
      end else if curAlias.Name <> oldAlias.Name then begin
        Result := CHANGED_RECORD;
      end;
      if Result = NO_CHANGE then begin
        if curAlias.SSN <> oldAlias.SSN then Result := CHANGED_RECORD;
      end;
    end;  
  end;
  
  function CompAlias(IEN : string; pAlias : tAlias; OldInfo : TPatientInfo) : integer;
  //format: s=IEN#, Object is ^tAlias  
  //Returns: NO_CHANGE = 1; NEW_RECORD = 2; DELETED_RECORD = 3; CHANGED_RECORD = 4;
  var i : integer;
      oldPAlias : tAlias;
  begin
    Result := NEW_RECORD;
    for i := 0 to OldInfo.AliasInfo.Count-1 do begin
      if OldInfo.AliasInfo.Strings[i] = IEN then begin
        oldPAlias := tAlias(OldInfo.AliasInfo.Objects[i]);
        Result :=  CompAliasRec(pAlias,oldPAlias);
        break;
      end;
    end;
  end;

  var i,j,AddCt : integer;  
      pAlias, tempPAlias : tAlias;
 
begin
  {if OldInfo = This Info, then remove entries}
  CompStrs(LName, OldInfo.LName);
  CompStrs(FName, OldInfo.FName);
  CompStrs(MName, OldInfo.MName);
  CompStrs(CombinedName, OldInfo.CombinedName);
  CompStrs(Prefix, OldInfo.Prefix);
  CompStrs(Suffix, OldInfo.Suffix);
  CompStrs(Degree, OldInfo.Degree);
  CompStrs(DOB, OldInfo.DOB);
  CompStrs(SSNum, OldInfo.SSNum);
  
  CompStrs(AddressLine1, OldInfo.AddressLine1);
  CompStrs(AddressLine2, OldInfo.AddressLine2);
  CompStrs(AddressLine3, OldInfo.AddressLine3);
  CompStrs(City, OldInfo.City);
  CompStrs(State, OldInfo.State);
  CompStrs(Zip4, OldInfo.Zip4);
  CompStrs(TempAddressLine1, OldInfo.TempAddressLine1);
  CompStrs(TempAddressLine2, OldInfo.TempAddressLine2);
  CompStrs(TempAddressLine3, OldInfo.TempAddressLine3);
  CompStrs(TempCity, OldInfo.TempCity);
  CompStrs(TempState, OldInfo.TempState);
  CompStrs(TempZip4, OldInfo.TempZip4);
  CompStrs(TempStartingDate , OldInfo.TempStartingDate );
  CompStrs(TempEndingDate , OldInfo.TempEndingDate );
  CompStrs(ConfidentalAddressLine1, OldInfo.ConfidentalAddressLine1);
  CompStrs(ConfidentalAddressLine2, OldInfo.ConfidentalAddressLine2);
  CompStrs(ConfidentalAddressLine3, OldInfo.ConfidentalAddressLine3);
  CompStrs(ConfidentalCity, OldInfo.ConfidentalCity);
  CompStrs(ConfidentalState, OldInfo.ConfidentalState);
  CompStrs(ConfidentalZip, OldInfo.ConfidentalZip);
  CompStrs(ConfidentalStartingDate , OldInfo.ConfidentalStartingDate );
  CompStrs(ConfidentalEndingDate , OldInfo.ConfidentalEndingDate );

  CompBoolUC(BadAddress, OldInfo.BadAddress);
  CompBoolUC(TempAddressActive, OldInfo.TempAddressActive);
  CompBoolUC(ConfAddressActive, OldInfo.ConfAddressActive);
  
  CompStrs(PhoneNumResidence, OldInfo.PhoneNumResidence);
  CompStrs(PhoneNumWork, OldInfo.PhoneNumWork);
  CompStrs(PhoneNumCell, OldInfo.PhoneNumCell);
  CompStrs(PhoneNumTemp, OldInfo.PhoneNumTemp);
  CompStrs(Sex, OldInfo.Sex);

  //Compare Aliases
  //format: s=IEN#, Object is ^tAlias  

  //first, see which entries in OldInfo are deleted in CurInfo.
  for i := 0 to OldInfo.AliasInfo.Count-1 do begin
    pAlias := tAlias(OldInfo.AliasInfo.Objects[i]);
    if CompAlias(OldInfo.AliasInfo.Strings[i], pAlias, self) = NEW_RECORD then begin
      //here we have an entry in OldInfo, not in CurInfo, so must represent a Delete
      //This needs to be posted to server with old IEN and @ symbol
      tempPAlias := tAlias.Create;
      tempPAlias.Name := '@';
      AliasInfo.AddObject(OldInfo.AliasInfo.Strings[i],tempPAlias);          
    end; 
  end;  
  
  AddCt := 0;
  //First, see which entries in New PatientInfo are new, or unchanged.
  for i := 0 to AliasInfo.Count-1 do begin
    pAlias := tAlias(AliasInfo.Objects[i]);
    if (pAlias=nil) then continue;
    if pAlias.Name= '@' then continue;   //skip those marked as deleted from OldInfo
    case CompAlias(AliasInfo.Strings[i], pAlias, OldInfo) of
      NO_CHANGE      : begin  //delete unchanged data (no need to repost to server)
                         pAlias.Destroy;
                         AliasInfo.Strings[i] :='<@>';  //mark for deletion below
                       end;
      NEW_RECORD :     begin  //mark as +1, +2 etc IEN
                         AddCt := AddCt + 1;
                         AliasInfo.Strings[i] := '+' + IntToStr(AddCt);
                       end;
     CHANGED_RECORD :  begin end;  // do nothing, leave changes in place
    end; {case}  
  end;

  for i := AliasInfo.Count-1 downto 0 do begin
    if AliasInfo.Strings[i] = '<@>' then AliasInfo.Delete(i);
  end;
  
  
end;


//=========================================================
//=========================================================
//=========================================================

procedure TfrmPtDemoEdit.GetPtInfo;
var  tempINI : TMemINIFile;  //I do this to make dealing with hash table read easier
     i,index : integer;
     IEN, Key,Value,s : string;
     pAlias : tAlias;
     
begin
  FServerPatientInfo.Clear;  
  tempINI := TMemINIFile.Create('xxx.ini');
  
  RPCBrokerV.remoteprocedure := 'TMG GET PATIENT DEMOGRAPHICS';
  RPCBrokerV.param[0].value := Patient.DFN;   RPCBrokerV.param[0].ptype := literal;
  RPCBrokerV.Call;
  
  with FServerPatientInfo do begin
    //Store results in a hash table for easier random access
    //Don't store Alias info in hash table, put directly into AliasInfo stringlist
    for i := 0 to RPCBrokerV.Results.Count-1 do begin
      s := RPCBrokerV.Results.Strings[i];
      if Pos('ALIAS',s)=0 then begin
        Key := piece(s,'=',1);
        Value := piece(s,'=',2);
        tempINI.WriteString('DATA',Key,Value);
      end else begin
        IEN := piece(s,' ',2);
        if StrToInt(IEN)>MaxAliasIEN then MaxAliasIEN := StrToInt(IEN);
        index := AliasInfo.IndexOf(IEN);
        if index <0 then begin
          pAlias := tAlias.Create;  //AliasInfo will own these.
          AliasInfo.AddObject(IEN,pAlias);
        end else begin
          pAlias := tAlias(AliasInfo.Objects[index]);
        end;
        if Pos('NAME=',s)>0 then begin
          pAlias.Name := piece(s,'=',2);
        end else if Pos('SSN=',s)>0 then begin
          pAlias.SSN := piece(s,'=',2);
        end;  
      end;
    end;
    LName:=tempINI.ReadString('DATA','LNAME','');
    FName:=tempINI.ReadString('DATA','FNAME','');
    MName:=tempINI.ReadString('DATA','MNAME','');
    CombinedName:=tempINI.ReadString('DATA','COMBINED_NAME','');
    Prefix:=tempINI.ReadString('DATA','PREFIX','');
    Suffix:=tempINI.ReadString('DATA','SUFFIX','');
    Degree:=tempINI.ReadString('DATA','DEGREE','');
    DOB:= tempINI.ReadString('DATA','DOB','');
    Sex:= tempINI.ReadString('DATA','SEX','');
    SSNum:= tempINI.ReadString('DATA','SS_NUM','');
    AddressLine1:= tempINI.ReadString('DATA','ADDRESS_LINE_1','');
    AddressLine2:= tempINI.ReadString('DATA','ADDRESS_LINE_2','');
    AddressLine3:= tempINI.ReadString('DATA','ADDRESS_LINE_3','');
    City:= tempINI.ReadString('DATA','CITY','');
    State:= tempINI.ReadString('DATA','STATE','');
    Zip4:= tempINI.ReadString('DATA','ZIP4','');
    BadAddress:= BoolUC(tempINI.ReadString('DATA','BAD_ADDRESS','')<>'');
    TempAddressLine1:= tempINI.ReadString('DATA','TEMP_ADDRESS_LINE_1','');
    TempAddressLine2:= tempINI.ReadString('DATA','TEMP_ADDRESS_LINE_2','');
    TempAddressLine3:= tempINI.ReadString('DATA','TEMP_ADDRESS_LINE_3','');
    TempCity:= tempINI.ReadString('DATA','TEMP_CITY','');
    TempState:=tempINI.ReadString('DATA','TEMP_STATE','');
    TempZip4:= tempINI.ReadString('DATA','TEMP_ZIP4','');
    TempStartingDate :=tempINI.ReadString('DATA','TEMP_STARTING_DATE','');
    TempEndingDate := tempINI.ReadString('DATA','TEMP_ENDING_DATE','');
    TempAddressActive:= BoolUC(tempINI.ReadString('DATA','TEMP_ADDRESS_ACTIVE','')='YES');
    ConfidentalAddressLine1:= tempINI.ReadString('DATA','CONF_ADDRESS_LINE_1','');
    ConfidentalAddressLine2:= tempINI.ReadString('DATA','CONF_ADDRESS_LINE_2','');
    ConfidentalAddressLine3:= tempINI.ReadString('DATA','CONF_ADDRESS_LINE_3','');
    ConfidentalCity:= tempINI.ReadString('DATA','CONF_CITY','');
    ConfidentalState:= tempINI.ReadString('DATA','CONF_STATE','');
    ConfidentalZip:= tempINI.ReadString('DATA','CONF_ZIP','');
    ConfidentalStartingDate := tempINI.ReadString('DATA','CONG_STARTING_DATE','');
    ConfidentalEndingDate := tempINI.ReadString('DATA','CONF_ENDING_DATE','');
    ConfAddressActive:= BoolUC(tempINI.ReadString('DATA','CONF_ADDRESS_ACTIVE','')='YES');
    PhoneNumResidence:= tempINI.ReadString('DATA','PHONE_RESIDENCE','');
    PhoneNumWork:= tempINI.ReadString('DATA','PHONE_WORK','');
    PhoneNumCell:= tempINI.ReadString('DATA','PHONE_CELL','');
    PhoneNumTemp:= tempINI.ReadString('DATA','PHONE_TEMP','');
  end;
  FCurPatientInfo.Assign(FServerPatientInfo);
  tempINI.Free;  //I don't write out, so should never end up on disk.
end;

procedure TfrmPtDemoEdit.PostChangedInfo(PatientInfo : TPatientInfo);

  procedure CheckBUCPost(Title : string; Value : BoolUC);
  begin 
    if Value <> bucUnchanged then begin
      if Value = bucTrue then RPCBrokerV.Param[1].Mult['"'+Title+'"'] := 'YES';
      if Value = bucFalse then RPCBrokerV.Param[1].Mult['"'+Title+'"'] := 'NO';
    end;  
  end;

  procedure CheckPost(Title, Value : string);
  begin 
    if Value <> '' then RPCBrokerV.Param[1].Mult['"'+Title+'"'] := Value;
  end;
  
  var i : integer;
      pAlias : tAlias;
begin
  RPCBrokerV.remoteprocedure := 'TMG SET PATIENT DEMOGRAPHICS';
  RPCBrokerV.param[0].value := Patient.DFN;   RPCBrokerV.param[0].ptype := literal;

  RPCBrokerV.Param[1].PType := list;
  with PatientInfo do begin
    CheckPost('COMBINED_NAME',CombinedName);
    //CheckPost('LNAME', LName);    //Don't send because data is in COMBINED NAME
    //CheckPost('FNAME',FName);
    //CheckPost('MNAME',MName);
    //CheckPost('PREFIX',Prefix);
    //CheckPost('SUFFIX',Suffix);
    //CheckPost('DEGREE',Degree);
    CheckPost('DOB',DOB);
    CheckPost('SEX',Sex);
    CheckPost('SS_NUM',SSNum);
    CheckPost('ADDRESS_LINE_1',AddressLine1);
    CheckPost('ADDRESS_LINE_2',AddressLine2);
    CheckPost('ADDRESS_LINE_3',AddressLine3);
    CheckPost('CITY',City);
    CheckPost('STATE',State);
    CheckPost('ZIP4',Zip4);

    CheckPost('TEMP_ADDRESS_LINE_1',TempAddressLine1);
    CheckPost('TEMP_ADDRESS_LINE_2',TempAddressLine2);
    CheckPost('TEMP_ADDRESS_LINE_3',TempAddressLine3);
    CheckPost('TEMP_CITY',TempCity);
    CheckPost('TEMP_STATE',TempState);
    CheckPost('TEMP_ZIP4',TempZip4);
    CheckPost('TEMP_STARTING_DATE',TempStartingDate );
    CheckPost('TEMP_ENDING_DATE',TempEndingDate );
    CheckPost('CONF_ADDRESS_LINE_1',ConfidentalAddressLine1);
    CheckPost('CONF_ADDRESS_LINE_2',ConfidentalAddressLine2);
    CheckPost('CONF_ADDRESS_LINE_3',ConfidentalAddressLine3);
    CheckPost('CONF_CITY',ConfidentalCity);
    CheckPost('CONF_STATE',ConfidentalState);
    CheckPost('CONF_ZIP',ConfidentalZip);
    CheckPost('CONG_STARTING_DATE',ConfidentalStartingDate );
    CheckPost('CONF_ENDING_DATE',ConfidentalEndingDate );
    CheckPost('PHONE_RESIDENCE',PhoneNumResidence);
    CheckPost('PHONE_WORK',PhoneNumWork);
    CheckPost('PHONE_CELL',PhoneNumCell);
    CheckPost('PHONE_TEMP',PhoneNumTemp);

    case BadAddress of
      bucTrue:  RPCBrokerV.Param[1].Mult['"BAD_ADDRESS"'] := 'UNDELIVERABLE';
      bucFalse: RPCBrokerV.Param[1].Mult['"BAD_ADDRESS"'] := '@';
    end; {case}
    CheckBUCPost('TEMP_ADDRESS_ACTIVE', TempAddressActive);
    CheckBUCPost('CONF_ADDRESS_ACTIVE', ConfAddressActive);

    for i := 0 to AliasInfo.Count-1 do begin
      pAlias := tAlias(AliasInfo.Objects[i]);
      if (pAlias=nil) then continue;
      RPCBrokerV.Param[1].Mult['"ALIAS ' + AliasInfo.Strings[i] +  ' NAME"'] := pAlias.Name;
      if pAlias.Name <> '@' then begin
        RPCBrokerV.Param[1].Mult['"ALIAS ' + AliasInfo.Strings[i] +  ' SSN"'] := pAlias.SSN;
      end;
    end;

    RPCBrokerV.Call;
    if RPCBrokerV.Results.Strings[0]<>'1' then begin
      MessageDlg(RPCBrokerV.Results.Strings[0],mtError,[mbOK],0);
    end else begin
      ChangesMade := true;
    end;
  end;  
end;


procedure TfrmPtDemoEdit.ShowPtInfo(Patient : TPatientInfo);
var i : integer;
    pAlias : tAlias;
begin
  ProgNameChangeOccuring := true;
  With Patient do begin
    LNameEdit.Text := LName;
    FNameEdit.Text := FName;
    MNameEdit.Text := MName;
    CombinedNameEdit.Text := CombinedName;
    PrefixEdit.Text := Prefix;
    SuffixEdit.Text := Suffix;
    DegreeEdit.Text := Degree;
    DOBEdit.Text := DOB;
    SSNumEdit.Text := SSNum;
    if Sex='MALE' then SexComboBox.ItemIndex := 0 else SexComboBox.ItemIndex := 1;
    AliasComboBox.Items.Clear;
    for i := 0 to AliasInfo.Count-1 do begin
      pAlias := tAlias(AliasInfo.Objects[i]);
      if pAlias<>nil then begin
        AliasComboBox.Items.AddObject(pAlias.Name,pAlias);
      end;
    end;
    if AliasComboBox.Items.count>0 then begin
      AliasComboBox.ItemIndex := 0;
      SetAliasEnabled(true);
    end;
    ShowAliasInfo(Patient);
    
    PhoneNumGrp.ItemIndex := 0;
    PhoneNumGrpClick(self);
  end;  

  BadAddressCB.Visible := false;
  TempActiveCB.Visible := false;
  ProgNameChangeOccuring := false;
end;

procedure TfrmPtDemoEdit.ShowAliasInfo(Patient : TPatientInfo);
var i : integer;
    pAlias : tAlias;
begin
  i := AliasComboBox.ItemIndex;
  if i > -1 then begin
    if i < Patient.AliasInfo.Count then begin
      pAlias := tAlias(Patient.AliasInfo.Objects[i]);
    end else pAlias := nil;  
    ProgAliasChangeOccuring := true;
    if pAlias<>nil then begin
      AliasNameEdit.Text := pAlias.Name;
      AliasSSNEdit.Text := pAlias.SSN;    
    end else begin
      AliasNameEdit.Text := '';
      AliasSSNEdit.Text := '';    
    end;
    ProgAliasChangeOccuring := false;
  end;  
end;


procedure TfrmPtDemoEdit.AliasComboBoxChange(Sender: TObject);
var s : string;
    i : integer;
begin
  if ProgAliasChangeOccuring=false then begin
    i := AliasCombobox.ItemIndex;
    if i>-1 then begin
      s := AliasCombobox.Items.Strings[i];
      SetAliasEnabled(true);
    end else begin
      SetAliasEnabled(false);
    end;  
    ShowAliasInfo(FCurPatientInfo);
  end;  
end;

procedure TfrmPtDemoEdit.FormCreate(Sender: TObject);
begin
  FCurAliasEdit := -1;
  FCurPatientInfo := TPatientInfo.Create;
  FServerPatientInfo := TPatientInfo.Create;
  ProgAliasChangeOccuring  := false;
  ProgNameChangeOccuring := false;
  ProgPhoneChangeOccuring := false;
  ProgAddressChangeOccuring := false;
  MaxAliasIEN := 0;
  ChangesMade := false;
end;

procedure TfrmPtDemoEdit.FormDestroy(Sender: TObject);
begin
  FCurPatientInfo.Destroy;
  FServerPatientInfo.Destroy;
end;


procedure TfrmPtDemoEdit.AddressRGrpClick(Sender: TObject);
begin
  {fill in data for newly selected fields}
  ProgAddressChangeOccuring := true;
  TempActiveCB.Visible := false;
  BadAddressCB.Visible := false;
  ConfActiveCB.Visible := false;
  StartingDateLabel.Visible := false;
  StartingDateEdit.Visible := false;
  EndingDateLabel.Visible := false;
  EndingDateEdit.Visible := false;
  case AddressRGrp.ItemIndex of
    0 : begin //Permanant
          AddressLine1Edit.Text := FCurPatientInfo.AddressLine1;
          AddressLine2Edit.Text := FCurPatientInfo.AddressLine2;
          AddressLine3Edit.text := FCurPatientInfo.AddressLine3;
          CityEdit.text := FCurPatientInfo.City;
          StateComboBox.text := FCurPatientInfo.State;
          Zip4Edit.text := FCurPatientInfo.Zip4;
          BadAddressCB.Visible := true;
          BadAddressCB.Checked := (FCurPatientInfo.BadAddress=bucTrue);
        end;
    1 : begin //temp
          AddressLine1Edit.Text := FCurPatientInfo.TempAddressLine1;
          AddressLine2Edit.Text := FCurPatientInfo.TempAddressLine2;
          AddressLine3Edit.Text := FCurPatientInfo.TempAddressLine3;
          CityEdit.Text := FCurPatientInfo.TempCity;
          StateComboBox.Text := FCurPatientInfo.TempState;
          Zip4Edit.Text := FCurPatientInfo.TempZip4;
          StartingDateEdit.Text := FCurPatientInfo.TempStartingDate ;
          EndingDateEdit.Text := FCurPatientInfo.TempEndingDate ;
          StartingDateLabel.Visible := true;
          StartingDateEdit.Visible := true;
          EndingDateLabel.Visible := true;
          EndingDateEdit.Visible := true;
          TempActiveCB.Visible := true;
          TempActiveCB.Checked := (FCurPatientInfo.TempAddressActive=bucTrue);
        end;
    2 : begin //confidental
          AddressLine1Edit.Text := FCurPatientInfo.ConfidentalAddressLine1;
          AddressLine2Edit.Text := FCurPatientInfo.ConfidentalAddressLine2;
          AddressLine3Edit.Text := FCurPatientInfo.ConfidentalAddressLine3;
          CityEdit.Text := FCurPatientInfo.ConfidentalCity;
          StateComboBox.Text := FCurPatientInfo.ConfidentalState;
          Zip4Edit.Text := FCurPatientInfo.ConfidentalZip;
          StartingDateEdit.Text := FCurPatientInfo.ConfidentalStartingDate ;
          EndingDateEdit.Text := FCurPatientInfo.ConfidentalEndingDate ;
          StartingDateLabel.Visible := true;
          StartingDateEdit.Visible := true;
          EndingDateLabel.Visible := true;
          EndingDateEdit.Visible := true;
          ConfActiveCB.Visible := true;
          ConfActiveCB.Checked := (FCurPatientInfo.ConfAddressActive=bucTrue);
        end;
  end; {case}
  ProgAddressChangeOccuring := false;
end;

procedure TfrmPtDemoEdit.PhoneNumGrpClick(Sender: TObject);
begin
  ProgPhoneChangeOccuring := true;
  case PhoneNumGrp.ItemIndex of
    0 : begin //Residence
          PhoneNumEdit.Text := FCurPatientInfo.PhoneNumResidence;
        end;
    1 : begin //work
          PhoneNumEdit.Text := FCurPatientInfo.PhoneNumWork;
        end;
    2 : begin //cell
          PhoneNumEdit.Text := FCurPatientInfo.PhoneNumCell;
        end;
    3 : begin //temp
          PhoneNumEdit.Text := FCurPatientInfo.PhoneNumTemp;
        end;
   else begin
          PhoneNumEdit.Text := '';
        end;    
  end; {case}
  ProgPhoneChangeOccuring := false;
end;

procedure TfrmPtDemoEdit.DelAliasBtnClick(Sender: TObject);
var i, j : integer;
    pAlias : tAlias;
begin
  i := AliasComboBox.ItemIndex;
  if i > -1 then begin
    pAlias := tAlias(AliasComboBox.Items.Objects[i]);
    if pAlias<>nil then begin
      for j := 0 to FCurPatientInfo.AliasInfo.Count-1 do begin
        if FCurPatientInfo.AliasInfo.Objects[j] = pAlias then begin
          FCurPatientInfo.AliasInfo.Delete(j);
          SetModified(true);
          pAlias.Free;
          AliasComboBox.Items.Delete(i);
          //AliasComboBox.ItemIndex := 0;
          //ShowAliasInfo(FCurPatientInfo);
          ProgAliasChangeOccuring:= true;
          AliasNameEdit.Text:='';
          AliasSSNEdit.Text:='';
          ProgAliasChangeOccuring:= false;
          break;
        end;
      end;
    end;
  end;
  if AliasCombobox.Items.Count=0 then begin
    AliasNameEdit.Text:='';
    AliasSSNEdit.Text:='';          
    SetAliasEnabled(false);
    AliasCombobox.Text := '';
  end;
end;

procedure TfrmPtDemoEdit.FormShow(Sender: TObject);
begin
  GetPtInfo(FServerPatientInfo);
  FCurPatientInfo.Assign(FServerPatientInfo);
  ShowPtInfo(FCurPatientInfo);  
end;

procedure TfrmPtDemoEdit.CombinedNameEditChange(Sender: TObject);
var LName,FName,MName : string;
begin 
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.CombinedName := CombinedNameEdit.Text; 
    SetModified(true);
    if CombinedNameEdit.Text <> CombinedName then begin
      ProgNameChangeOccuring := true;
      NameParts(CombinedNameEdit.Text, LName, FName, MName);
      LNameEdit.Text := LName;
      FNameEdit.Text := FName;
      MNameEdit.Text := MName;
      ProgNameChangeOccuring := false;
    end;
  end;    
end;

procedure TfrmPtDemoEdit.LNameEditChange(Sender: TObject);
begin 
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.LName := LNameEdit.Text; 
    SetModified(true);
    CombinedNameEdit.Text := CombinedName;
  end;
end;

procedure TfrmPtDemoEdit.FNameEditChange(Sender: TObject);
begin 
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.FName := FNameEdit.Text; 
    SetModified(true);
    CombinedNameEdit.Text := CombinedName;
  end  
end;

procedure TfrmPtDemoEdit.MNameEditChange(Sender: TObject);
begin 
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.MName := MNameEdit.Text;
    SetModified(true);
    CombinedNameEdit.Text := CombinedName;
  end;
end;

procedure TfrmPtDemoEdit.PrefixEditChange(Sender: TObject);
begin
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.Prefix := PrefixEdit.Text;
    SetModified(true);
  end;
end;

procedure TfrmPtDemoEdit.SuffixEditChange(Sender: TObject);
begin
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.Suffix := SuffixEdit.Text;
    SetModified(true);
    CombinedNameEdit.Text := CombinedName;
  end;
end;

procedure TfrmPtDemoEdit.DOBEditChange(Sender: TObject);
begin
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.DOB := DOBEdit.Text;
    SetModified(true);
  end;
end;

procedure TfrmPtDemoEdit.SSNumEditChange(Sender: TObject);
begin
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.SSNum := SSNumEdit.Text;
    SetModified(true);
  end;
end;

procedure TfrmPtDemoEdit.EMailEditChange(Sender: TObject);
begin
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.EMail := EMailEdit.Text;
    SetModified(true);
  end;
end;

procedure TfrmPtDemoEdit.SexComboBoxChange(Sender: TObject);
begin
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.Sex := SexComboBox.Text;
    SetModified(true);
  end;
end;

procedure TfrmPtDemoEdit.DegreeEditChange(Sender: TObject);
begin  
  if ProgNameChangeOccuring = false then begin
    FCurPatientInfo.Degree := DegreeEdit.Text;
    SetModified(true);
  end;  
end;

procedure TfrmPtDemoEdit.AddressLine1EditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    0 : FCurPatientInfo.AddressLine1 := AddressLine1Edit.Text;
    1 : FCurPatientInfo.TempAddressLine1 := AddressLine1Edit.Text;
    2 : FCurPatientInfo.ConfidentalAddressLine1 := AddressLine1Edit.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.AddressLine2EditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    0 : FCurPatientInfo.AddressLine2 := AddressLine2Edit.Text;
    1 : FCurPatientInfo.TempAddressLine2 := AddressLine2Edit.Text;
    2 : FCurPatientInfo.ConfidentalAddressLine2 := AddressLine2Edit.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.AddressLine3EditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    0 : FCurPatientInfo.AddressLine3 := AddressLine3Edit.Text;
    1 : FCurPatientInfo.TempAddressLine3 := AddressLine3Edit.Text;
    2 : FCurPatientInfo.ConfidentalAddressLine3 := AddressLine3Edit.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.CityEditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    0 : FCurPatientInfo.City  := CityEdit.Text;
    1 : FCurPatientInfo.TempCity := CityEdit.Text;
    2 : FCurPatientInfo.ConfidentalCity := CityEdit.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.Zip4EditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    0 : FCurPatientInfo.Zip4   := Zip4Edit.Text;
    1 : FCurPatientInfo.TempZip4 := Zip4Edit.Text;
    2 : FCurPatientInfo.ConfidentalZip := leftstr(Zip4Edit.Text,5);
  end;  {case}
end;

procedure TfrmPtDemoEdit.StateComboBoxChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    0 : FCurPatientInfo.State := StateComboBox.Text;
    1 : FCurPatientInfo.TempState := StateComboBox.Text;
    2 : FCurPatientInfo.ConfidentalState := StateComboBox.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.TempActiveCBClick(Sender: TObject);
begin 
  FCurPatientInfo.TempAddressActive := BoolUC(TempActiveCB.Checked); 
  if ProgAddressChangeOccuring = false then SetModified(true);
end;

procedure TfrmPtDemoEdit.ConfActiveCBClick(Sender: TObject);
begin 
  FCurPatientInfo.ConfAddressActive := BoolUC(ConfActiveCB.Checked); 
  if ProgAddressChangeOccuring = false then SetModified(true);
end;

procedure TfrmPtDemoEdit.BadAddressCBClick(Sender: TObject);
begin 
  FCurPatientInfo.BadAddress := BoolUC(BadAddressCB.Checked); 
  if ProgAddressChangeOccuring = false then SetModified(true);
end;

procedure TfrmPtDemoEdit.PhoneNumEditChange(Sender: TObject);
begin
  if ProgPhoneChangeOccuring = false then SetModified(true);
  Case PhoneNumGrp.ItemIndex of
    0 : FCurPatientInfo.PhoneNumResidence  := PhoneNumEdit.Text;
    1 : FCurPatientInfo.PhoneNumWork := PhoneNumEdit.Text;
    2 : FCurPatientInfo.PhoneNumCell := PhoneNumEdit.Text;
    3 : FCurPatientInfo.PhoneNumTemp := PhoneNumEdit.Text;
  end;  {case}

end;

procedure TfrmPtDemoEdit.StartingDateEditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    1 : FCurPatientInfo.TempStartingDate := StartingDateEdit.Text;
    2 : FCurPatientInfo.ConfidentalStartingDate := StartingDateEdit.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.EndingDateEditChange(Sender: TObject);
begin
  if ProgAddressChangeOccuring = false then SetModified(true);
  Case AddressRGrp.ItemIndex of
    1 : FCurPatientInfo.TempEndingDate := EndingDateEdit.Text;
    2 : FCurPatientInfo.ConfidentalEndingDate := EndingDateEdit.Text;
  end;  {case}
end;

procedure TfrmPtDemoEdit.AliasNameEditChange(Sender: TObject);
var i : integer;
    pAlias : tAlias;
    tempB : boolean;
begin
  if ProgAliasChangeOccuring=false then begin
    i := AliasComboBox.ItemIndex;
    if i > -1 then begin
      pAlias := tAlias(AliasComboBox.Items.Objects[i]);
      if pAlias<>nil then begin
        pAlias.Name := AliasNameEdit.Text;
        AliasComboBox.Items.Strings[i]:= AliasNameEdit.Text;
        AliasComboBox.Text := AliasNameEdit.Text;
        tempB := ProgAliasChangeOccuring;
        ProgAliasChangeOccuring:=true;
        AliasComboBox.ItemIndex := i;
        ProgAliasChangeOccuring:=tempB;
        SetModified(true);
      end;
    end;
  end;  
end;


procedure TfrmPtDemoEdit.AliasSSNEditChange(Sender: TObject);
var i : integer;
    pAlias : tAlias;
begin
  if ProgAliasChangeOccuring=false then begin
    i := AliasComboBox.ItemIndex;
    if i > -1 then begin
      pAlias := tAlias(AliasComboBox.Items.Objects[i]);
      if pAlias<>nil then begin
        pAlias.SSN := AliasSSNEdit.Text;      
        SetModified(true);
      end;
    end;
  end;  
end;

procedure TfrmPtDemoEdit.SetAliasEnabled(value : boolean);
begin
  AliasNameEdit.Enabled := value;
  AliasSSNEdit.Enabled := value;
  if value=true then begin
    AliasNameEdit.Color := clWindow;
    AliasSSNEdit.Color := clWindow;
  end else begin
    AliasNameEdit.Color := clInactiveBorder;
    AliasSSNEdit.Color := clInactiveBorder;
  end;
end;


function TfrmPtDemoEdit.CombinedName : string;
begin
  Result := '';
  Result := FNameEdit.Text;
  if MNameEdit.Text <> '' then Result := Result + ' ' + MNameEdit.Text;
  if SuffixEdit.Text <> '' then Result := Result + ' ' + SuffixEdit.Text;
  if Result <> '' then Result := ',' + Result;
  Result := LNameEdit.Text + Result;
end;

procedure TfrmPtDemoEdit.NameParts(CombinedName: string; var LName, FName, MName : string);
var tempS : string;
begin
  LName := piece(CombinedName,',',1);
  tempS := piece(CombinedName,',',2);
  FName := piece(tempS,' ',1);
  MName := piece(tempS,' ',2,16);
end;


procedure TfrmPtDemoEdit.ApplyBtnClick(Sender: TObject);
var TempPatientInfo : tPatientInfo;
begin
  TempPatientInfo := tPatientInfo.Create;
  TempPatientInfo.Assign(FCurPatientInfo);
  TempPatientInfo.RemoveUnchanged(FServerPatientInfo);
  PostChangedInfo(TempPatientInfo);
  TempPatientInfo.Destroy;
  SetModified(false);
end;

procedure TfrmPtDemoEdit.SetModified(value : boolean);
begin
  FCurPatientInfo.Modified := value;
  ApplyBtn.Enabled := FCurPatientInfo.Modified;
end;

procedure TfrmPtDemoEdit.AddAliasBtnClick(Sender: TObject);
var pAlias : tAlias;
    IEN : string;
begin
    pAlias := tAlias.Create;
    if FCurPatientInfo.AliasInfo.count>0 then begin
      IEN := FCurPatientInfo.AliasInfo.Strings[FCurPatientInfo.AliasInfo.count-1];
    end else begin
      IEN := IntToStr(MaxAliasIEN);
    end;  
    MaxAliasIEN := StrToInt(IEN)+1;
    IEN := IntToStr(MaxAliasIEN);
    FCurPatientInfo.AliasInfo.AddObject(IEN,pAlias);
    SetModified(true);
    AliasCombobox.Items.AddObject('<Edit New Alias>',pAlias);
    //pAlias.Name := '<Edit New Alias>';
    //AliasCombobox.Items.Add(ADD_NEW_ALIAS);
    AliasCombobox.ItemIndex := AliasCombobox.Items.Count-1;    
    SetAliasEnabled(true);
    ShowAliasInfo(FCurPatientInfo);
end;

procedure TfrmPtDemoEdit.OKBtnClick(Sender: TObject);
begin
  if FCurPatientInfo.Modified = true then begin
    case MessageDlg('Apply Changes?',mtConfirmation,mbYesNoCancel,0) of
      mrYes : begin
                ApplyBtnClick(Sender);    
                frmPtDemoEdit.ModalResult := mrOK;  //closes form                                        
              end;
      mrNo : begin
               if ChangesMade = false then frmPtDemoEdit.ModalResult := mrCancel // closes form, signal no need for refresh
               else frmPtDemoEdit.ModalResult := mrOK; // closes form.
             end;  
      mrCancel :  frmPtDemoEdit.ModalResult := mrNone; //do nothing
    end; {case}  
  end else begin
    if ChangesMade = false then frmPtDemoEdit.ModalResult := mrCancel // closes form, signal no need for refresh
    else frmPtDemoEdit.ModalResult := mrOK; // closes form.
  end;  
end;

procedure TfrmPtDemoEdit.CancelBtnClick(Sender: TObject);
begin
  if FCurPatientInfo.Modified = true then begin
    case MessageDlg('Cancel Changes?',mtConfirmation,mbYesNoCancel,0) of
      mrYes : frmPtDemoEdit.ModalResult := mrCancel;  //closes form                                        
      mrNo : frmPtDemoEdit.ModalResult := mrNone;  //do nothing
      mrCancel :  frmPtDemoEdit.ModalResult := mrNone; //do nothing
    end; {case}  
  end else begin
    frmPtDemoEdit.ModalResult := mrCancel; // closes form.
  end;  

end;



end.
